/**************************** FM2504.c **********************************/
/* Copyright 2005/02/28 Aeolus Development				*/
/* All rights reserved.							*/
/*									*/
/* Redistribution and use in source and binary forms, with or without	*/
/* modification, are permitted provided that the following conditions	*/
/* are met:								*/
/* 1. Redistributions of source code must retain the above copyright	*/
/*   notice, this list of conditions and the following disclaimer.	*/
/* 2. Redistributions in binary form must reproduce the above copyright	*/
/*   notice, this list of conditions and the following disclaimer in the*/
/*   documentation and/or other materials provided with the 		*/
/*   distribution.							*/
/* 3. The name of the Aeolus Development or its contributors may not be	*/
/* used to endorse or promote products derived from this software 	*/
/* without specific prior written permission.				*/
/*									*/
/* THIS SOFTWARE IS PROVIDED BY THE AEOULUS DEVELOPMENT "AS IS" AND ANY	*/
/* EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE	*/
/* IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR	*/
/* PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE AEOLUS DEVELOPMENT BE	*/
/* LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR	*/
/* CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF	*/
/* SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR 	*/
/* BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,*/
/* WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE */
/* OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, 	*/
/* EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.			*/
/*									*/
/*  Sub-device support for Ramtron FM2504. Requires an SPI driver as 	*/
/* parent to perform the actual communication.			 	*/
/* Note:  All the actual routines are private to this module.  The only */
/* element publically visible is the creation routine.			*/
/************************************************************************/
/*
*   TLIB revision history:
*   1 FM2504.c 31-Dec-2005,15:29:00,`RADSETT' Original version.
*   TLIB revision history ends.
*/
#include <stdio.h>
#include <errno.h>
#include <fcntl.h>
#include <string.h>
#include "FM2504.h"
#include "lpc_ioctl.h"

	/**** Local Macros ****/

#define FM2504_LENGTH	((_off_t)512)	/*  Size of FRAM in bytes.	*/


	/**** Local Types ****/

	/**** Local Prototypes ****/

static int fm2504_blocking_setup( struct _reent *r, const int *p, FM2504_INSTANCE *us);
static int fm2504_close( struct _reent *r, int file, const SUB_DEVICE_INFO *info);
static int fm2504_init( struct _reent *r, const SUB_DEVICE_INFO *info);
static int fm2504_ioctl( struct _reent *r, int file, unsigned long request, void *ptr, const SUB_DEVICE_INFO *info);
static int fm2504_open(	struct _reent *r, const char *name, int o_flags, int o_mode, const SUB_DEVICE_INFO *info);
static _ssize_t fm2504_read ( struct _reent *r, int file, void *ptr, size_t len, const SUB_DEVICE_INFO *info);
static int fm2504_seek( struct _reent *r, struct device_seek *off, FM2504_INSTANCE *us);
static _ssize_t fm2504_write( struct _reent *r, int file, const void *ptr, size_t len, const SUB_DEVICE_INFO *info);
static int process_queue( struct _reent *r, const SUB_DEVICE_INFO *info);

	/**** Local Variables ****/

	/*  FM2504_line - desired communication characteristics of the	*/
	/* device.							*/
static const struct spi_param FM2504_line = {
     SPI_CPHA_EDGE1,		/*  Data clocked on first edge.		*/
     SPI_CPOL_HIGH,		/*  Clock is positive (first edge up).	*/
     SPI_BIT_DIR_MSBF,		/*  Send MSB first.			*/
     20000000uL			/*  20MHz max speed.			*/
     };

	/*  init_command - byte string sent to device to ensure that it	*/
	/* is in a known state at startup.				*/
static const unsigned char init_command[] = {
    0x6,			/*  Write Enable.			*/
    0x1, 			/*  Write status register command.	*/
    0x0,			/*  Argument to command.		*/
    0x4				/*  Write Disable.			*/
    };

/********************* process_queue ************************************/
/*  process_queue	-- Process the command queue, return as soon as	*/
/* possible.  If driver is blocking on I/O that will be done by the 	*/
/* caller.								*/
/*  struct _reent *r	-- re-entrancy structure, used by newlib to 	*/
/*			support multiple threads of operation.		*/
/*  SUB_DEVICE_INFO *info -- Per instance information, used to provide	*/
/*			information on a per instance basis.		*/
/*  Returns 0 if all commands are finished.				*/
static int process_queue( 
    struct _reent *r, 
    const SUB_DEVICE_INFO *info)	/*  Per instance information.	*/
{
 FM2504_INSTANCE *us;
 struct device_select sel;
 int val;
 struct spi_param spi_line;	/*  A temporary copy used since it must	*/
    				/* be writeable.			*/
 struct buffer_param buf_mode;
 int block_mode;

 us = info->device_info;	/*  Get preserved information about the	*/
 				/* device we are manipulating.		*/

	/*  First check to see if we are waiting for any bytes. This 	*/
	/* will only happen for a buffered device.  By reading only 	*/
	/* bytes that are available we avoid blocking at this stage and	*/
	/* let the caller do any blocking that is needed.		*/
 if(us->read_chars) {
    size_t num_waiting;

	/*  Find out if any characters are waiting to be read.  If call	*/
	/* returns an error abort.  That probably means a bad 		*/
	/* configuration.						*/
    if( ioctl( us->parent_file,  UART_CHAR_WAITING, &num_waiting) != 0) {
         return -1;
         }

	/* Read in any waiting bytes.					*/
    if( num_waiting > (size_t)0) { 
		/*  There is a buffer waiting, read bytes into the 	*/
		/* buffer and advance the buffer position.		*/
	 if( us->read_buffer != 0) {
			/*  Read waiting bytes.  If an error occurs 	*/
			/* send error code back to the caller.		*/
	      if( info->parent->read( r, us->parent_file, us->read_buffer, 
	           num_waiting, info->parent->info) < (_ssize_t)0){
		   return -1;
	           }
	      us->read_buffer += num_waiting;
	      }

		/*  No buffer, read each byte and discard,		*/
	 else {
	      char temp;
	      size_t i;

	      for( i = 0; i < num_waiting; i++) {
			/*  Read each waiting byte.  If an error occurs	*/
			/* send error code back to the caller.		*/
		   if( info->parent->read( r, us->parent_file, &temp, (size_t)1, 
		        info->parent->info) < (_ssize_t)0){
		        return -1;
		        }
		   }
	      }

		/*  Now reduce the number of bytes we are waiting for.	*/
	 us->read_chars -= num_waiting;
	 }
    }

 if( us->read_chars != 0) {	/*  Check to see if we've read all the 	*/
    return 1;			/* bytes we are waiting for, if not	*/
    }				/* return indicating there is still work*/
				/* left to do.				*/

 val = r->_errno;	/*  Save errno for restoration, since some 	*/
 			/* operations may effect it in the normal	*/
 			/* course of action.				*/

	/*  We need to check two things here to determine if the next	*/
	/* command should be executed.  First we check the state of the	*/
	/* command queue, a position of -1 indicates that this is a new	*/
	/* command sequence to be started.  If that is not the case we	*/
	/* then check to see if the previous command has finished.  If	*/
	/* one of these conditions is true we can proceed to execute 	*/
	/* the next command.  It is tempting to rely only on the second */
	/* check but the file number retained with the device will not	*/
	/* likely be valid on the first command queue setup and may not	*/
	/* remain a constant number across multiple calls in any case.	*/
 
 if( us->command_queue_position != -1) {
    if( info->parent->write( r, us->parent_file, 0, 0uL, info->parent->info) != 0) {
         r->_errno = val;	/*  Restore errno.			*/
         return 1;		/*  Return indicating more to be done.	*/
         }
    }


 us->command_queue_position++;		/*  Advance to the next command.*/

	/*  Perform the requested command.				*/
 switch( us->q[us->command_queue_position].cmd) {

	/*  CMD_OPEN - The first command in the queue.  Attempt to open	*/
	/* and set up the parent device for use by the subsequent	*/
	/* commands.							*/
    case CMD_OPEN:
		/*  Attempt to open the parent device.  Open as read & 	*/
		/* write since all activity requires both. 		*/
         us->parent_file = info->parent->open( r, "", O_RDWR, 0666, /*lint !e960 octal constant */
              info->parent->info);

		/*  If we can't open, the usual cause will be another	*/
		/* child device is using it at the moment.  Back off 	*/
		/* command and return to allow other device to finish.	*/
	 if( us->parent_file < 0) {	
              us->command_queue_position--;
	      break;
	      }

		/*  Set up the line characteristics for the device.	*/
		/* An error probably indicates the parent device is not */
		/* an SPI device, return to caller.			*/
         spi_line = FM2504_line;
	 if( info->parent->ioctl( r, us->parent_file, SPI_SETUP, &spi_line, 
	      info->parent->info) != 0){
	      return -1;
	      }

		/*  Set up buffering mode.  Set to zero buffering if 	*/
		/* possible.						*/
	 us->zero_buffer = 0;	/*  Indicate buffered until proved 	*/
	 			/* otherwise.				*/
	 buf_mode.type = BUFFER_ZERO;
	 if( info->parent->ioctl( r, us->parent_file, BUFFER_SETUP, &buf_mode, 
	      info->parent->info)== 0) {
	      if( buf_mode.type == BUFFER_ZERO) {
		   us->zero_buffer = 1;	/*  Successfully set to	zero 	*/
		   }			/* buffer mode.			*/
	      }

		/*  Set to non-blocking if possible.  Done for 		*/
		/* consistency rather than out of necessity.		*/
	 block_mode = BLOCKING_IO_NO;
	 (void)info->parent->ioctl( r, us->parent_file, BLOCKING_SETUP, 
	      &block_mode, info->parent->info);

	 	/*  Finally select the device.  If an error occurs	*/
		/* simply return it to the caller.			*/
	 sel.device_number = info->device_number;
	 sel.select_action = SELECT_ACTION_SEL;
	 if( info->parent->ioctl( r, us->parent_file, DEVICE_SELECT, &sel, 
	      info->parent->info) != 0) {
	      return -1;
	      }
	 break;

	/*  CMD_NONE - Indicates the end of the command list, clean up	*/
	/* and finish.							*/
    case CMD_NONE:

		/*  Deselect the device.  If an error occurs simply	*/
		/* return it to the caller.				*/
         sel.device_number = info->device_number;
         sel.select_action = SELECT_ACTION_DESEL;
         if( info->parent->ioctl( r, us->parent_file, DEVICE_SELECT, &sel, 
              info->parent->info) != 0) {
	      return -1;
	      }

		/*  Close parent, this will allow other subdevices	*/
		/* access to the parent device.  If error occurs simply	*/
		/* return it to the caller.				*/
	 if( info->parent->close( r, us->parent_file, info->parent->info) != 0) {
	      return -1;
	      }
	 us->parent_file = -1;

		/*  Set first queue entry to indicate an empty queue.	*/
	 us->q[0].cmd = CMD_NONE;
         us->command_queue_position = 0;
	 return 0;	/*  Return indicating all done.			*/

	/*  CMD_READ - Perform a read operation.  For this device a 	*/
	/* read operation is done with a dummy write with the written	*/
	/* values having no significance.  In order to be sure that we	*/
	/* have a buffer we can write we will write out the buffer we	*/
	/* will be reading into since we know that a) it must exist in	*/
	/* the memory space and b) it must be large enough.  If we	*/
	/* wanted the output to be predictable we could preset the	*/
	/* buffer before using it.					*/
    case CMD_READ:
		
		/*  Zero buffer case.  First set up the read, then 	*/
		/* perform the write to do the transfer.  In case of	*/
		/* error simply return to the caller.			*/
	 if( us->zero_buffer != 0) {
	      if( info->parent->read( r, us->parent_file, 
	           us->q[us->command_queue_position].ptr, 0uL, info->parent->info) < 
	           (_ssize_t)0) {

		   return -1;
		   }
	      if( info->parent->write( r, us->parent_file, 
	           us->q[us->command_queue_position].ptr, 
	           us->q[us->command_queue_position].len, info->parent->info) < 
	           (_ssize_t)0) {

		   return -1;
		   }
	      }

		/*  Buffered case.  First start the write, then queue	*/
		/* up the read for subsequent calls.  In case of an	*/
		/* error simply return to the caller.			*/
	 else {
	      if( info->parent->write( r, us->parent_file, 
	           us->q[us->command_queue_position].ptr, 
	           us->q[us->command_queue_position].len, info->parent->info) < 
	           (_ssize_t)0) {

		   return -1;
		   }
	      us->read_chars = us->q[us->command_queue_position].len;
	      us->read_buffer = us->q[us->command_queue_position].ptr;
	      }
	 break;

	/*  CMD_WRITE - Perform a write operation.  For this device a	*/
	/* write operation is done with a dummy read to discard the read*/
	/* values which have no significance.				*/
    case CMD_WRITE:

		/*  Zero buffer case. Set up a read with a null pointer,*/
		/* the parent driver will automattically discard the 	*/
		/* read bytes.  Then send the write request.  In the 	*/
		/* case of an error simply return to the caller.	*/
	 if( us->zero_buffer != 0) {
	      if( info->parent->read( r, us->parent_file, 0, 0uL, 
	           info->parent->info) < (_ssize_t)0) {

		   return -1;
		   }
	      if( info->parent->write( r, us->parent_file, 
	           us->q[us->command_queue_position].ptr, 
	           us->q[us->command_queue_position].len, info->parent->info) < 
	           (_ssize_t)0) {

		   return -1;
		   }
	      }

		/*  Buffered case.  First start the write, then queue	*/
		/* up the read for subsequent calls.  Using a null 	*/
		/* pointer for the read buffer tells the subsequent read*/
		/* operations to discard the read in bytes.  In the case*/
		/* of an error simply return to the caller.		*/
	 else {
	      if( info->parent->write( r, us->parent_file, 
	           us->q[us->command_queue_position].ptr, 
	           us->q[us->command_queue_position].len, info->parent->info) < 
	           (_ssize_t)0) {

	           return -1;
		   }
	      us->read_chars = us->q[us->command_queue_position].len;
	      us->read_buffer = 0;
	      }
	 break;
    }

 r->_errno = val;	/*  Restore errno.				*/
 return 1;		/*  Return indicating more to be done.		*/
}

/********************* fm2504_init **************************************/
/*  fm2504_init		-- Setup the SPI pins and data structures.	*/
/* Should be called only once.						*/
/*  struct _reent *r	-- re-entrancy structure, used by newlib to 	*/
/*			support multiple threads of operation.		*/
/*  SUB_DEVICE_INFO *info -- Per instance information, used to provide	*/
/*			information on a per instance basis.		*/
/*  Returns 0 if successful.						*/
static int fm2504_init( 
    struct _reent *r, 
    const SUB_DEVICE_INFO *info)	/*  Per instance information.	*/
{
 FM2504_INSTANCE *us;

 us = info->device_info;	/*  Get preserved information about the	*/
 				/* device we are manipulating.		*/

	/*  Set up the command queue for initialization.		*/

	/*  First open the parent device.				*/
 us->q[us->command_queue_position].ptr = 0;
 us->q[us->command_queue_position].len = 0;
 us->q[us->command_queue_position].cmd = CMD_OPEN;
 us->command_queue_position++;

	/*  Second write the initialzation command sequence.		*/
 us->q[us->command_queue_position].ptr = init_command;	/*lint !e605 increase in pointer capability*/
 us->q[us->command_queue_position].len = sizeof( init_command);
 us->q[us->command_queue_position].cmd = CMD_WRITE;
 us->command_queue_position++;

	/*  Third, close the parent device, we are done.		*/
 us->q[us->command_queue_position].ptr = 0;
 us->q[us->command_queue_position].len = 0;
 us->q[us->command_queue_position].cmd = CMD_NONE;
 us->command_queue_position = -1;

	/* Initialization always blocks until complete.			*/
 while( process_queue(r, info) != 0) {
    }

 return 0;
}


/********************* fm2504_blocking_setup ****************************/
/*  fm2504_blocking_setup -- Setup the drivers blocking response. 	*/
/*  struct _reent *r	-- re-entrancy structure, used by newlib to 	*/
/*			support multiple threads of operation.		*/
/*  const unsigned int *p -- requested blocking mode.			*/
/*  FM2504_INSTANCE *us -- Per instance information, used to provide	*/
/*			information on a per instance basis.		*/
/*  Returns 0 if successful						*/
static int fm2504_blocking_setup( 
    struct _reent *r, 
    const int *p,
    FM2504_INSTANCE *us)	/*  Per instance information.	*/
{

	/*  Not much to do other than note the mode.			*/
 switch( *p) {
    case BLOCKING_IO_YES:
    case BLOCKING_IO_NO:
         us->blocking = *p;
	 break;

	/*  Unknown mode, set to default and flag as an error.		*/
    default:
         us->blocking = BLOCKING_IO_NO;
         r->_errno = ENOSYS;	/*  Request not understood.		*/
	 return -1;
    }

 return 0;	/* Success.						*/
 }

/********************* fm2504_seek **************************************/
/*  fm2504_seek		-- Sets current device offset.			*/
/*  struct _reent *r	-- re-entrancy structure, used by newlib to 	*/
/*			support multiple threads of operation.		*/
/*  struct device_seek *off -- offset in device to position to for next	*/
/*			read or write.  This will be updated with the	*/
/*			actual position set to on return.		*/
/*  FM2504_INSTANCE *us -- Per instance information, used to provide	*/
/*			information on a per instance basis.		*/
/*  Returns 0 if successful.						*/
static int fm2504_seek( 
    struct _reent *r, 
    struct device_seek *off,
    FM2504_INSTANCE *us)	/*  Per instance information.	*/
{

 switch( off->dir) {

	/*  SEEK_SET - Position from beginning of device, simply set 	*/
	/* position to requested value.					*/
    case SEEK_SET:
         us->device_offset = off->off;
	 break;

	/*  SEEK_END - Position from end of device, add requested 	*/
	/* value to size of the device.					*/
    case SEEK_END:
         us->device_offset = FM2504_LENGTH + off->off;
	 break;

	/*  SEEK_CUR - Position from current position, add requested 	*/
	/* value to current position.					*/
    case SEEK_CUR:
         us->device_offset += off->off;
	 break;

    default:			/*  Unknown command, complain.		*/
         r->_errno = EINVAL;
	 return -1;
    }

 us->device_offset %= FM2504_LENGTH;	/*  Wrap around device length.	*/
 off->off = us->device_offset;	/*  Return current device position. 	*/
 return 0;			/*  Success.				*/
}

/********************* fm2504_ioctl *************************************/
/*  fm2504_ioctl -- Provides ioctl functions for fm2504.  Essentially a	*/
/* cracker function, it figures	out what is being requested and calls 	*/
/* the appropriate routine to do it.					*/
/*  struct _reent *r	-- re-entrancy structure, used by newlib to 	*/
/*			support multiple threads of operation.		*/
/*  int file		-- number referring to the open file. Generally	*/
/*			obtained from a corresponding call to open.	*/
/*			Only one file number (0) is supported.		*/
/*  unsigned long request -- simple ordinal indicating what the 	*/
/*			requested action is.				*/
/*  void *ptr		-- pointer to data to read and or write to	*/
/*			perform request.				*/
/*  SUB_DEVICE_INFO *info -- Per instance information, used to provide	*/
/*			information on a per instance basis.  Only one	*/
/*			instance of this device is allowed.		*/
/*  Returns 0 if successful.  						*/
static int fm2504_ioctl(
    struct _reent *r, 
    int file, 
    unsigned long request, 
    void *ptr,
    const SUB_DEVICE_INFO *info)/*  Per instance information.		*/
{
 if( file != 0) {		/* Only one device.			*/
    r->_errno = EBADF;		/* Bad file number.			*/
    return -1;
    }

 switch( request) {

	/*  DEVICE_INIT - Initializes device and associated device info.*/
    case DEVICE_INIT:
         return fm2504_init( r, info);

	/*  DEVICE_SEEK - Move current position to a specific device	*/
	/* location.							*/
    case DEVICE_SEEK:
         return fm2504_seek( r, ptr, info->device_info);

    case BLOCKING_SETUP:	/*  Do we block on I/O?			*/
         return fm2504_blocking_setup( r, ptr, info->device_info);

    default:
         r->_errno = ENOSYS;	/*  Request not understood.		*/
	 return -1;
    }
}


/************************** fm2504_read *********************************/
/*  Reads from fm2504 using parent device.				*/
/*  struct _reent *r	-- re-entrancy structure, used by newlib to 	*/
/*			support multiple threads of operation.		*/
/*  int file		-- number referring to the open file. Generally	*/
/*			obtained from a corresponding call to open.	*/
/*			Only one file number (0) is supported.		*/
/*  void *ptr		-- memory area to place read bytes into. 	*/
/*  size_t len		-- maximum number of bytes to read.		*/
/*  SUB_DEVICE_INFO *info -- Per instance information, used to provide	*/
/*			information on a per instance basis.		*/
/*  Note:  will only return a single byte on every call. Blocks until	*/
/* that byte is ready. 							*/
/*  Returns number of bytes read. (_ssize_t)-1 on error.		*/
static _ssize_t fm2504_read (
    struct _reent *r, 		/*  Re-entrancy structure, used to make	*/
				/* thread safe.				*/
    int file,			/*  File handle.  Used to distinguish	*/
				/* multiple instances.  		*/
    void *ptr,		 	/*  Where to place data.		*/
    size_t len,			/*  Max data to read.			*/
    const SUB_DEVICE_INFO *info)/*  Per instance information.		*/
{
 FM2504_INSTANCE *us;

 if( file != 0) {		/* Only one device.			*/
    r->_errno = EBADF;		/* Bad file number.			*/
    return (_ssize_t)-1;
    }

	/*  Check to see if previous command is still being processed.	*/
	/* If so indicate that the device is busy.			*/
 if( process_queue(r, info) != 0) { 
    r->_errno = EBUSY;
    return (_ssize_t)-1;
    }

 us = info->device_info;	/*  Get preserved information about the	*/
 				/* device we are manipulating.		*/

	/*  Build up the command queue for a read.  Start at position 0.*/
 us->command_queue_position = 0;

	/*  First open the parent device.				*/
 us->q[us->command_queue_position].ptr = 0;
 us->q[us->command_queue_position].len = 0;
 us->q[us->command_queue_position].cmd = CMD_OPEN;
 us->command_queue_position++;

	/*  Set up local buffer for read starting from current position	*/
 us->buf[0] = 0x3 | ((((int)us->device_offset & 0x100) != 0)? 0x8 : 0);
 us->buf[1] = (int)us->device_offset & 0xFF;

	/*  Write read command to device.				*/
 us->q[us->command_queue_position].ptr = us->buf;
 us->q[us->command_queue_position].len = (size_t)2;
 us->q[us->command_queue_position].cmd = CMD_WRITE;
 us->command_queue_position++;

	/*  Read requested # bytes from the device.			*/
 us->q[us->command_queue_position].ptr = ptr;
 us->q[us->command_queue_position].len = len;
 us->q[us->command_queue_position].cmd = CMD_READ;
 us->command_queue_position++;

	/*  All done, close the parent device.				*/
 us->q[us->command_queue_position].cmd = CMD_NONE;
 us->command_queue_position = -1;

 us->device_offset += (_off_t)len;	/*  Set up position to match 	*/
 					/* what it will be when done.	*/
 us->device_offset %= FM2504_LENGTH;	/*  Wrap around device length.	*/

 (void)process_queue(r, info);		/* Start executing commands.	*/

 if( us->blocking != 0) {		/*  If set up for blocking,	*/
    while( process_queue( r, info)!= 0){/* Wait for command to finish.	*/
         }
    }

 return (_ssize_t)len;
}

/************************** fm2504_write ********************************/
/*  Writes to fm2504 using parent device.				*/
/*  struct _reent *r	-- re-entrancy structure, used by newlib to 	*/
/*			support multiple threads of operation.		*/
/*  int file		-- number referring to the open file. Generally	*/
/*			obtained from a corresponding call to open.	*/
/*			Only one file number (0) is supported.		*/
/*  const void *ptr	-- memory area to place read bytes from. 	*/
/*  size_t len		-- maximum number of bytes to write.		*/
/*  SUB_DEVICE_INFO *info -- Per instance information, used to provide	*/
/*			information on a per instance basis.  Only one	*/
/*			instance of this device is allowed.		*/
/*  Returns number of bytes written. (_ssize_t)-1 on error.		*/
static _ssize_t fm2504_write(
    struct _reent *r,  		/*  Re-entrancy structure, used to make	*/
				/* thread safe.				*/
    int file, 			/*  File handle.  Used to distinguish	*/
				/* multiple instances.  		*/
    const void *ptr,		/*  Pointer to data to write.		*/
    size_t len,			/*  Amount of data to write.		*/
    const SUB_DEVICE_INFO *info)/*  Per instance information.		*/
{
 FM2504_INSTANCE *us;

 if( file != 0) {		/* Only one device.			*/
    r->_errno = EBADF;		/* Bad file number.			*/
    return (_ssize_t)-1;
    }

 us = info->device_info;	/*  Get preserved information about the	*/
 				/* device we are manipulating.		*/

	/*  Check to see if previous command is still being processed.	*/
	/* If so indicate that the device is busy.			*/
 if( process_queue( r, info)!= 0) {
    r->_errno = EBUSY;
    return (_ssize_t)-1;
    }

	/*  Build up the command queue for a write. Start at position 0.*/
 us->command_queue_position = 0;

	/*  First open the parent device.				*/
 us->q[us->command_queue_position].ptr = 0;
 us->q[us->command_queue_position].len = 0;
 us->q[us->command_queue_position].cmd = CMD_OPEN;
 us->command_queue_position++;

	/*  Set up local buffer for write starting from current position*/
 us->buf[0] = 0x2 | ((((int)us->device_offset & 0x100) != 0)? 0x8 : 0);
 us->buf[1] = (int)us->device_offset & 0xFF;/*??*/

	/*  Write the write command to device.				*/
 us->q[us->command_queue_position].ptr = us->buf;
 us->q[us->command_queue_position].len = (size_t)2;
 us->q[us->command_queue_position].cmd = CMD_WRITE;
 us->command_queue_position++;

	/*  Write the data out to the device.				*/
 us->q[us->command_queue_position].ptr = ptr;	/*lint !e605 increase in pointer capability*/
 us->q[us->command_queue_position].len = len;
 us->q[us->command_queue_position].cmd = CMD_WRITE;
 us->command_queue_position++;

	/*  All done, close the parent device.				*/
 us->q[us->command_queue_position].cmd = CMD_NONE;
 us->command_queue_position = -1;

 us->device_offset += (_off_t)len;	/*  Set up position to match 	*/
 					/* what it will be when done.	*/
 us->device_offset %= FM2504_LENGTH;	/*  Wrap around device length.	*/

 (void)process_queue(r, info);		/* Start executing commands.	*/

 if( us->blocking != 0) {		/*  If set up for blocking,	*/
    while( process_queue(r, info)!= 0){	/*  Wait for command to finish.	*/
         }
    }

 return (_ssize_t)len;			/* Number of bytes written.	*/
}

/************************** fm2504_open *********************************/
/*  Opens fm2504.							*/
/*  struct _reent *r	-- re-entrancy structure, used by newlib to 	*/
/*			support multiple threads of operation.		*/
/*  const char *name	-- name of file/device to open. Since sys 	*/
/*			supports no sub devices this should be an empty	*/
/*			string.						*/
/*  int flags		-- Flags to control open.  Not used at the 	*/
/*			moment.						*/
/*  int mode		-- Mode to open in.  Not used at the moment.	*/
/*  SUB_DEVICE_INFO *info -- Per instance information, used to provide	*/
/*			information on a per instance basis.		*/
/*  Returns file number >= 0 if successful.  Otherwise the error code 	*/
/* may be found in errno.						*/
/*lint -esym(715,o_flags, o_mode, info) not referenced.			*/
static int fm2504_open(	
    struct _reent *r,		/*  Re-entrancy structure, used to make	*/
				/* thread safe.				*/
    const char *name, 		/*  Name to open.			*/
    int o_flags,		/*  Flags to control open.		*/
				/* Read, write binary etc...		*/
				/* See flags.c for values generated by	*/
				/* newlib.				*/
    int o_mode,			/*  Mode to open in.  This is a 	*/
    				/* security or permissions value.  	*/
    				/* Newlib uses the classic 0666 for all */
    				/* fopens.  See fopen.c			*/
    const SUB_DEVICE_INFO *info)/*  Per instance information.		*/
{

	/*  Check against null pointer.  Also no sub-devices available	*/
	/* so make sure we aren't asked to open one.			*/
 if( (name == 0) || (*name != '\0')) {
    r->_errno = ENOENT;		/* No such file or directory.		*/
    return( -1);
    }
 return( 0);			/*  Always sub-handle 0.  Note we never	*/
				/* fail on this open.			*/
}

/************************** fm2504_close ********************************/
/*  Close the fm2504.							*/
/*  struct _reent *r	-- re-entrancy structure, used by newlib to 	*/
/*			support multiple threads of operation.		*/
/*  int file		-- number referring to the open file. Generally	*/
/*			obtained from a corresponding call to open.	*/
/*			Only one file number (0) is supported.		*/
/*  SUB_DEVICE_INFO *info -- Per instance information, used to provide	*/
/*			information on a per instance basis.  Only one	*/
/*			instance of this device is allowed.		*/
/*  Returns 0 if successful.  Otherwise the error code may be found in 	*/
/* errno.								*/
static int fm2504_close(
    struct _reent *r, 		/*  Re-entrancy structure, used to make	*/
				/* thread safe.				*/
    int file,			/*  File/device sub handle.		*/
    const SUB_DEVICE_INFO *info)/*  Per instance information.		*/
{

 if( file != 0) {		/* Only one device.			*/
    r->_errno = EBADF;		/* Bad file number.			*/
    return -1;
    }
 return( 0);			/* Always succeeds.			*/
}

/************************** fm2504 **************************************/
/*  Device table entry used to add this device.				*/
static const struct device_table_entry fm2504 = {
	"dummy",			/*  Device name placeholder.	*/
	fm2504_open,  			/*  Open method.		*/
	fm2504_close,			/*  Close method.		*/
	fm2504_read,			/*  Read method.		*/
	fm2504_write,			/*  Write method.		*/
	fm2504_ioctl,			/*  Ioctl method.		*/
	0 };				/*  Per-instance data placeholder.*/

/************************** CreateFM2504 ********************************/
/*  Given the appropriate structures initialize them to create a driver	*/
/* that can be added to an SPI driver to control an FM2504.		*/
/*  NOTE:  The parameters passed by pointer MUST last as long as the	*/
/* device driver, they are not copied.					*/
int CreateFM2504(  
    unsigned int device_number, 	/*  Device number to select the	*/
					/* particular FM2504.		*/
    const char *name, 			/*  Name to give the device.	*/
    FM2504_INSTANCE *instance, 		/*  Structure to contain FM2504 */
    					/* instance specific data.	*/
    struct device_table_entry *dt, 	/*  Device driver table.	*/
    SUB_DEVICE_INFO *inf)		/*  General instance specific 	*/
    					/* data.			*/
{
int i;

memcpy( dt, &fm2504, sizeof( fm2504));
dt->info = inf;
dt->name = name;

inf->device_info = instance;
inf->device_number = device_number;

	/*  Initialize the command queue.				*/
 for ( i = 0; i < CMD_QUEUE_LENGTH; i++) {
    instance->q[i].cmd = CMD_NONE;
    instance->q[i].ptr = 0;
    instance->q[i].len = 0;
    }
 instance->command_queue_position = 0;

 instance->read_chars = 0;	/*  Nothing waiting to be read.		*/
 instance->read_buffer = 0;

 instance->parent_file = -1;	/*  File number of the parent device.	*/
 instance->device_offset = 0;	/*  Where in the device current read 	*/
    				/* and write operations should occur.	*/
 instance->zero_buffer = 0;	/*  Initially assume parent doesn't use	*/
				/* zero_buffering.			*/
 instance->blocking = BLOCKING_IO_YES;	/* Block on I/O.		*/

	/*  Zero temporary buffer.					*/
 for( i = 0; i < (int)(sizeof( instance->buf)/sizeof(instance->buf[0])); i++) {
    instance->buf[i] = 0;
    }
 return 0;
}
