/******************************************************************************/
/*  lpc_blink_switch_irq                                                      */
/*  LED Flasher - Timer Interrupt example for arm-elf-gcc                     */
/*  Lights LED on Startup and flashs LED when button is pressed.              */
/******************************************************************************/
/*  Inspired by a sample application from Keil Elektronik                     */
/*  A lot of information has been found in a sample from R O Software.        */
/******************************************************************************/
/*  Sample for WinARM by M.Thomas <eversmith@heizung-thomas.de>               */
/******************************************************************************/

#include "lpc210x_gnuarm.h"
#include "config.h"
#include "VIClowlevel.h"

#include "timer.h"

// olimex LPC-P2106: one led on P0.7 (active low)
#define LEDPIN  7
#define LEDMASK (1<<LEDPIN);
// olimex LPC-P2106: one switch on P0.31 (active low)
#define SWPIN 	31

#define WAIT100MS  10
#define WAIT1S     100

static void systemInit(void)
{
	// --- enable and connect the PLL (Phase Locked Loop) ---
	// a. set multiplier and divider
	SCB_PLLCFG = MSEL | (1<<PSEL1) | (0<<PSEL0);
	// b. enable PLL
	SCB_PLLCON = (1<<PLLE);
	// c. feed sequence
	SCB_PLLFEED = PLL_FEED1;
	SCB_PLLFEED = PLL_FEED2;
	// d. wait for PLL lock (PLOCK bit is set if locked)
	while (!(SCB_PLLSTAT & (1<<PLOCK)));
	// e. connect (and enable) PLL
	SCB_PLLCON = (1<<PLLE) | (1<<PLLC);
	// f. feed sequence
	SCB_PLLFEED = PLL_FEED1;
	SCB_PLLFEED = PLL_FEED2;
	
	// --- setup and enable the MAM (Memory Accelerator Module) ---
	// a. start change by turning of the MAM (redundant)
	MAM_MAMCR = 0;	
	// b. set MAM-Fetch cycle to 3 cclk as recommended for >40MHz
	MAM_MAMTIM = MAM_FETCH;
	// c. enable MAM 
	MAM_MAMCR = MAM_MODE;

	// --- set VPB speed ---
	SCB_VPBDIV = VPBDIV_VAL;

    // --- map INT-vector ---
	#if defined(RAM_RUN)
	  SCB_MEMMAP = MEMMAP_USER_RAM_MODE;
	#elif defined(ROM_RUN)
	  SCB_MEMMAP = MEMMAP_USER_FLASH_MODE;
	#else
	#error RUN_MODE not defined!
	#endif
}

static void gpioInit(void)
{
	GPIO_IOSET  = (1<<LEDPIN);	// set Bit = LED off (active low)
	GPIO_IODIR |= (1<<LEDPIN);	// define LED-Pin as output
	GPIO_IODIR &= ~(1<<SWPIN);	// define Switch-Pin as input
}

static void wait (unsigned long delay) 
{
  unsigned long i;

  i = timeval + delay;
  while ( i != timeval);           
}

int main(void) 
{
	gpioInit();  
	
	GPIO_IOCLR = LEDMASK; 	// enable LED - "life sign"

	systemInit();			// PLL, MAM etc.
	
	init_timer();
	
	enableIRQ(); 
	
	wait(WAIT1S);
	GPIO_IOSET = LEDMASK; 	// disable LED
	wait(WAIT1S);

	while(1) {
		if (GPIO_IOPIN & (1<<SWPIN)) {	// true if button released (active low)
			GPIO_IOCLR = LEDMASK; // enable LED
		} 
		else {
		  GPIO_IOSET = LEDMASK; // disable LED
		  wait(WAIT100MS);
		  GPIO_IOCLR = LEDMASK; // enable LED
		  wait(WAIT100MS);
		}
	}
}
