/******************************************************************************/
/*  lpc_blink_switch_irq LPC2129                                              */
/*  LED Flasher - Timer Interrupt example for arm-elf-gcc                     */
/*  Lights LEDs on Startup and Flashs LEDs when Button1 is pressed.           */
/******************************************************************************/
/*  Inspired by a sample application from Keil Elektronik                     */
/*  A lot of information has been found in a sample from R O Software.        */
/******************************************************************************/
/*  Sample for WinARM by M.Thomas <eversmith@heizung-thomas.de>               */
/*  http://www.siwawi.arubi.uni-kl.de/avr_projects                            */
/******************************************************************************/

#include "lpc21xx_keil.h"
#include "config.h"
#include "VIClowlevel.h"

#include "timer.h"

// olimex LPC-P2129: buttons on P0.10/P0.11 (active low)
#define BUT1PIN 	10
#define BUT2PIN 	11
// olimex LPC-P2129: LEDs on P0.12/P0.13 (active low)
#define LED1PIN  	12
#define LED1BIT (1<<LED1PIN)
#define LED2PIN  	13
#define LED2BIT (1<<LED2PIN)

#define WAIT100MS  10
#define WAIT1S     100

static void systemInit(void)
{
	// --- enable and connect the PLL (Phase Locked Loop) ---
	// a. set multiplier and divider
	PLLCFG = MSEL | (1<<PSEL1) | (0<<PSEL0);
	// b. enable PLL
	PLLCON = (1<<PLLE);
	// c. feed sequence
	PLLFEED = PLL_FEED1;
	PLLFEED = PLL_FEED2;
	// d. wait for PLL lock (PLOCK bit is set if locked)
	while (!(PLLSTAT & (1<<PLOCK)));
	// e. connect (and enable) PLL
	PLLCON = (1<<PLLE) | (1<<PLLC);
	// f. feed sequence
	PLLFEED = PLL_FEED1;
	PLLFEED = PLL_FEED2;
	
	// --- setup and enable the MAM (Memory Accelerator Module) ---
	// a. start change by turning of the MAM (redundant)
	MAMCR = 0;	
	// b. set MAM-Fetch cycle to 3 cclk as recommended for >40MHz
	MAMTIM = MAM_FETCH;
	// c. enable MAM 
	MAMCR = MAM_MODE;

	// --- set VPB speed ---
	VPBDIV = VPBDIV_VAL;

    // --- map INT-vector ---
	#if defined(RAM_RUN)
	  MEMMAP = MEMMAP_USER_RAM_MODE;
	#elif defined(ROM_RUN)
	  MEMMAP = MEMMAP_USER_FLASH_MODE;
	#else
	#error RUN_MODE not defined!
	#endif
}

static void gpioInit(void)
{
	IODIR0 |= (1<<LED1PIN)|(1<<LED2PIN); // define LED-Pins as outputs
	IOSET0 = (1<<LED1PIN)|(1<<LED2PIN); // set Bits = LEDs off (active low)
	IODIR0 &= ~((1<<BUT1PIN)|(1<<BUT2PIN));// define Button-Pins as inputs
}

static void wait (unsigned long delay)  
{
  unsigned long i;

  i = timeval + delay;
  while ( i != timeval);           
}

int main(void) 
{
	gpioInit();  
	
	IOCLR0 = ( LED1BIT | LED2BIT ); 	// enable LED1 & 2 - "life sign"

	systemInit();			// PLL, MAM etc.
	
	init_timer();
	
	enableIRQ(); 
	
	wait(WAIT1S);
	IOSET0 = LED1BIT; 	// disable LED1
	IOCLR0 = LED2BIT; 	// enable LED2
	wait(WAIT1S);
	IOSET0 = LED2BIT; 	// disable LED2
	IOCLR0 = LED1BIT; 	// enable LED1
	wait(WAIT1S);

	while(1) {
		if (IOPIN0 & (1<<BUT1PIN)) {	// true if button released (active low)
			IOCLR0 = (LED1BIT | LED2BIT); // enable LEDs
		} 
		else {
		  IOSET0 = LED1BIT; // disable LED1
		  IOCLR0 = LED2BIT; // enable LED2
		  wait(WAIT100MS);
		  IOCLR0 = LED1BIT; // enable LED1
		  IOSET0 = LED2BIT; // disable LED2
		  wait(WAIT100MS);
		}
	}
	
	return 0; /* never reached */
}
