# UBB::FileHandler
#
# Assists in control of I/O using UBB::FileHandle objects
# Also blame this one on Charles Capps.

package UBB::FileHandler;
use strict;
no strict("refs");
use Fcntl qw(:DEFAULT :flock);
import UBBCGI::Carp qw(fatalsToBrowser set_message);

BEGIN {
	use vars qw($VERSION @ISA);

	@ISA = qw(UBB::CommonElements);

	unless(defined &SIXSIXSIX) {
		use constant SIXSIXSIX => 0666;
		use constant SEVENSEVENSEVEN => 0777;
	} # end unless
	umask(0000);

	$VERSION = 0.13;

	$| = 1;
	#open(STDERR, ">>/net/ubb622-development/cache-8DN2JVNW/lock/filehandlelog2") or die $!;
	#my $oldfh = select(STDERR); $|=1; select($oldfh);
} # end BEGIN

sub new {
	my $protoobj = shift;
	my $class = ref($protoobj) || $protoobj;
	my $self  = {
		'CONFIG' => $_[0],			# %vars_config
		'DEBUG' => 0,				# debugging (warnings)
		'HANDLES' => {},			# hashref of UBB::FileHandler objs
		'UNIQ' => UBB::FileHandler->_random(16),# This session's UNIQID for the warnings
	};
	bless ($self, $class);

	if(!-d "$_[0]->{NonCGIPath}/cache-$_[0]->{cache_pw}") {
		mkdir("$_[0]->{NonCGIPath}/cache-$_[0]->{cache_pw}", 0777);
		chmod(0777,"$_[0]->{NonCGIPath}/cache-$_[0]->{cache_pw}");
	}
	if(!-d "$_[0]->{NonCGIPath}/cache-$_[0]->{cache_pw}/lock") {
		mkdir("$_[0]->{NonCGIPath}/cache-$_[0]->{cache_pw}/lock", 0777);
		chmod(0777,"$_[0]->{NonCGIPath}/cache-$_[0]->{cache_pw}/lock");
	}
	# Despite the fact that we never write any data to it, we need to open the lockfile in
	# readwrite mode... why?  Because some OSes (such as Solaris) see no reasaon at all why
	# you should be allowed to LOCK_EX a file you're not writing to.
	# Thanks to 'hexmode' from K5 for pointing this out.  :)
	my $file = ($_[1] || "global") . ".cgi";
	$self->{'LOCKFILE'} = new UBB::FileHandle($self, 'lock', 'readwrite', "/$file"),
	$self->{'LOCKFILE'}->relock('UN');	# no lock until we start handling files

	$self->warn("STARTING UP");

	return $self;
} # end sub


sub DESTROY {	# FileHandler go booooooooom!
	my $self = shift;

	my $number_of_open = scalar keys %{$self->{'HANDLES'}} or 0;
	$self->warn("Closing remaining open files ($number_of_open)");
	if($number_of_open > 0) {
		$self->warn("Filenames are: ");
		foreach my $opened (keys %{$self->{'HANDLES'}}) {
			$self->warn("Filename: $opened");
		} # end foreach
		$self->warn("End filenames.  Let's start closing them.");

		foreach my $opened (keys %{$self->{'HANDLES'}}) {
			$self->warn("Currently disposing of $opened");
			if($self->{'HANDLES'}->{$opened}->{'object'} == undef) {
				$self->warn(">>> Looks like I've lost track of $opened");
			}
			delete $self->{'HANDLES'}->{$opened};
			$self->warn("$opened now removed.");
		} # end foreach
	} # end if

	$self->warn("Opened files now closed.  Removing global lock.");
	$self->{'LOCKFILE'}->_close() if(ref($self->{'LOCKFILE'}) =~ m/UBB/);
	$self->warn("CLOSING DOWN");

	return;
} # end DESTROY


sub warn {
	my $self = shift;
	return 1 unless $self->{'DEBUG'} > 0;

	my $prepend = $self->_gettime() . "-$self->{'UNIQ'}, ";

	my $pusher;
	foreach my $line (@_) {
		$pusher .= $prepend . $line . "\n";
	} # end foreach

	print STDERR $pusher;

	return;
} # end warn


sub open {	# spawns a UBB::FileHandle and returns it
	my $self = shift;
	$self->warn("Opening file as type $_[1]");
	if(($_[1] =~ m/write/) && ($_[1] !~ m/readwrite2/)) {
		# Writer but not readwrite2 -> exclusive
		$self->{'LOCKFILE'}->relock('EX');
	} else {
		# Else shared
		$self->{'LOCKFILE'}->relock('SH') unless $self->{'LOCKFILE'}->lockmode() eq 'EX';
	} # end if

	my $file = UBB::FileHandle->new($self, @_);
	if(exists $self->{'HANDLES'}->{$file->fullpath}) {
		$self->warn("PANIC: I already opened " . $file->fullpath());
		$self->warn("PANIC: I can't have the same file opened twice!  That screws things up MAJORLY.");
		$self->warn("PANIC: Shutting self down.");
		die("UBB::FileHandler was asked to open the same file (" . $file->fullpath() . ") twice!  \nUBB::FileHandler already had that file open when another request came in.  \nIt looks like there's an error in the code somewhere.\n<br />\nHere's the backtrace to this call: " . $self->tracer() . "<br />\nHere's the previously opened file's backtrace: " . $self->{'HANDLES'}->{$file->fullpath}->{'tracer'});
	}
	$self->{'HANDLES'}->{$file->fullpath}->{'object'} = $file;
	$self->{'HANDLES'}->{$file->fullpath}->{'tracer'} = $self->tracer();

	return $file;
} # end sub

sub close {	# terminates the UBB::FileHandler object, sets the lockfile as required
	my $self = shift;
	my $file = shift;
	$file->_close();
	delete $self->{'HANDLES'}->{$file->fullpath()};

	$self->warn("Disposing of " . $file->fullpath());

	my $openhandles = scalar(keys %{$self->{'HANDLES'}});

	if($openhandles > 0) {
		$self->warn("Still have $openhandles handles open, deciding mode");
		my $mode = "";
		HANDLECHECK: foreach my $list (keys %{$self->{'HANDLES'}}) {
			my $object = $self->{'HANDLES'}->{$list}->{'object'};
			if($object eq undef) {
				$self->warn("Looks like I've lost track of '$list'");
				$self->warn("Removing from records.");
				delete $self->{'HANDLES'}->{$list};
				next HANDLECHECK;
			} # end if
			$mode .= $object->readmode();
		} # end foreach

		if($mode =~ m/write[^2]/) {	# any writer that's not a write2
			# I have handles open and at least one is writing
			$self->warn("Have writers open");
			$self->{'LOCKFILE'}->relock('EX');
		} else {
			$self->warn("Have readers open");
			# I have handles open, and there are NO writers
			$self->{'LOCKFILE'}->relock('SH');
		}
	} else {
		# I have no more handles open
		$self->warn("No more open handles, removing lock");
		$self->{'LOCKFILE'}->relock('UN');
	} # end if

	$self->warn("COMPLETELY disposed of " . $file->fullpath());

	return 1;
} # end sub

sub force_lock {
	$_[0]->{'LOCKFILE'}->relock("EX");
} # end force_lock

sub force_unlock {
	$_[0]->{'LOCKFILE'}->relock("UN");
} # end force_unlock

sub _gettime {	# time for #$self->warn
	my $self = shift;
	#use Time::HiRes;
	#return Time::HiRes::time();
	return scalar time();
} # end _gettime

sub tracer {	# tracer, copied here so we don't need to rely on ubb_lib
	my $self = shift;
	my $string = "";
	my $i = 0;
	while(@_ = caller($i)) {
		my $file = $_[1];
		$file =~ s!(.*)(\\|/)([^/\\]+)$!$3!;
		$string .= "\n Backtrace: $file:$_[2] -> sub $_[3] ";
		$i++;
	}
	return $string . "\n";
} # end tracer

90 + 9 == 99;
# $Id: ubb_lib_filehandler.cgi,v 1.9 2002/03/15 01:16:44 cvscapps Exp $
