# UBB::FileHandle
#
# Module to handle all direct I/O to disk by UBB.classic
# Blame this one on Charles Capps.

package UBB::FileHandle;
use strict;
no strict("refs");
use Fcntl qw(:DEFAULT :flock);
import UBBCGI::Carp qw(fatalsToBrowser set_message);

BEGIN {
	use vars qw($VERSION @ISA);

	@ISA = qw(UBB::CommonElements);

	unless(defined &SIXSIXSIX) {
		use constant SIXSIXSIX => 0666;
		use constant SEVENSEVENSEVEN => 0777;
	} # end unless
	umask(0000);

	$VERSION = 0.13;

	$| = 1;
	#open(STDERR, ">>/net/ubb622-development/cache-8DN2JVNW/lock/filehandlelog2") or die $!;
	#my $oldfh = select(STDERR); $|=1; select($oldfh);
} # end BEGIN


sub new {
	my $protoobj = shift;
	my $class = ref($protoobj) || $protoobj;
	my $parent = shift;
	$parent = bless($parent, "UBB::FileHandler");
	my $self  = {	# these are hard coded into the script and are intentionally NOT overridable...
		'PARENT' => $parent,		# UBB::FileHandler, or a subclass thereof
		'CONFIG' => $parent->{'CONFIG'},# %vars_config
		'DEBUG' => 0,			# debugging (warnings)
		'WRITER' => 0,			# ++ed during each write
		'NOTAFILE' => 0,		# not a real file (i.e. emulation of a file using RDBMS?)
		'BUFFER' => 1, 			# use stdio rather than system calls (this MIGHT be unsafe, but I can't prove it)
		'IMALOCK' => 0,			# is this a lock file?
	};
	bless ($self, $class);

	$self->{'UNIQ'} = $self->_random(6);	# our uniqid

	my $filehandlemess = $self->_random(16);
	my $filehandle = *{$filehandlemess};
	$self->{'HANDLE'} = \$filehandle;

	my $filetype = shift;

	# custom filetypes - these are currently not entirely implemented
	# there's simply too much code calling the pre-2.0 lib_files routines
	# that need to be converted over... funny how things change so much in
	# just a couple of weeks, eh?  We wanted lib_files to be the plugin
	# replacement, not these two files...

	# I wanted to get this fixed for 6.2.0, but it's not going to happen.
	# 6.3?  7.0?
	if($filetype eq 'cache') {     $self->_open_cache(@_) }		# SPECIAL
	elsif($filetype eq 'index') {  $self->_open_index(@_) }
	elsif($filetype eq 'lock') {   $self->_open_lock(@_) }		# SPECIAL
	elsif($filetype eq 'thread') { $self->_open_thread(@_) }	# SPECIAL
	elsif($filetype eq 'forumd') { $self->_open_forumd(@_) }	# SPECIAL
	elsif($filetype eq 'member') { $self->_open_member(@_) }	# SPECIAL
	elsif($filetype eq 'hash') {   $self->_open_file(@_) }
	elsif($filetype eq 'hashes') { $self->_open_file(@_) }
	elsif($filetype eq 'flat') {   $self->_open_file(@_) }
	elsif($filetype eq 'file') {   $self->_open_file(@_) }
	else { die "I don't know what filetype '$filetype' is.\nHere is the backtrace: " . $self->tracer() }

	return $self;
} # end sub


sub relock {
	my $self = shift;
	return if $self->{'NOTAFILE'} > 1;
	# Normally this would cause problems, but we can be assured that
	# if we are doing I/O like this, we have a LOCK_EX on the central
	# file.  Either way, we have to unlock before relocking, because
	# certain brain damaged OSes can't handle changing
	# the lock, i.e. Windows.

	$self->warn("About to switch from $self->{'LOCKMODE'} to $_[0]");
	if($self->{'LOCKMODE'} eq $_[0]) {
		$self->warn("Can't do that, already in $_[0]!");
		return;
	}

	if($_[0] eq 'UN') {
		# unlock
		flock($self->{'HANDLE'}, LOCK_UN) or $self->process_bang("Can't UNlock '" . $self->fullpath() . "'.", $!, $self->tracer());
		$self->{'LOCKMODE'} = 'UN';
	} elsif(($_[0] eq 'EX') && ($self->{'LOCKMODE'} ne 'EX')) {
		# exclusive lock
		if($self->{'LOCKMODE'} ne 'UN') {
			flock($self->{'HANDLE'}, LOCK_UN) or $self->process_bang("Can't UNlock '" . $self->fullpath . "'.", $!, $self->tracer());
		} 	# end if
		flock($self->{'HANDLE'}, LOCK_EX) or $self->process_bang("Can't EXlock '" . $self->fullpath . "'.", $!, $self->tracer());
		$self->{'LOCKMODE'} = 'EX';
	} elsif(($_[0] eq 'SH') && ($self->{'LOCKMODE'} ne 'SH')) {
		# shared lock
		if($self->{'LOCKMODE'} ne 'UN') {
			flock($self->{'HANDLE'}, LOCK_UN) or $self->process_bang("Can't UNlock '" . $self->fullpath . "'.", $!, $self->tracer());
		} 	# end if
		flock($self->{'HANDLE'}, LOCK_SH) or $self->process_bang("Can't SHlock '" . $self->fullpath . "'.", $!, $self->tracer());
		$self->{'LOCKMODE'} = 'SH';
	} # end if

	$self->warn("Relock done!");

	return 1;
} # end relock


sub print {	# calls the print handler method for this particular file
	my $self = shift;
	# warn is inside handler

	&{$self->{'PRINTHANDLER'}}($self, @_);
	return 1;
} # end print

sub readline {	# calls the readline method for this particular file
	my $self = shift;
	#$self->warn("Readline called") unless $self->fullpath =~ m/memberslist\.cgi/;
	my $tempvar = &{$self->{'READLINEHANDLER'}}($self, @_);
	#$self->warn("Read " . length($tempvar) . " bytes") unless $self->fullpath =~ m/memberslist\.cgi/;
	return $tempvar;
} # end readline

sub readfile {	# calls the readfile method for this particular file
	my $self = shift;
	$self->warn("Readfile called");
	my $tempvar = &{$self->{'READFILEHANDLER'}}($self, @_);
	$self->warn("Read " . length($tempvar) . " bytes");
	return $tempvar;
} # end readline

sub readfile_asarray {	# calls the readfile method for this particular file, but returns an array
	my $self = shift;
	$self->warn("Readfile As Array called");
	my $tempvar = &{$self->{'READFILEHANDLER'}}($self, @_);
	$self->warn("Read " . length($tempvar) . " bytes");
	return split(/\n/, $tempvar);
} # end readline

sub truncate {	# calls the truncate method for this particular file
	my $self = shift;
	&{$self->{'TRUNCATEHANDLER'}}($self, @_);
} # end subs

sub readfile_filtered_asarrayref { 	# returns an arrayref containing a file
	my $self = shift;		# run though a user-defined filter sub
	my $filter = shift;		# sans blank lines.  Undef == eof
	my $returnage = [];

	$self->warn("readfile_filtered_asarrayref called.");

	while($_ = $self->readline) {
		chomp;
		my $fl = &{$filter}($_);
		push(@{ $returnage }, $fl) if($fl);
	} # end while

	$self->warn("readfile_filtered_asarrayref: returning a " . scalar(@{$returnage}) . " element array.");

	return $returnage;
} # end readfile_asfilteredarrayref

sub readfile_filtered_ashashref { 	# returns a hashref based on input data,
	my $self = shift;		# run though a user-defined filter sub
	my $filter = shift;
	my $returnage = {};

	$self->warn("readfile_filtered_ashashref called.");

	while($_ = $self->readline) {
		chomp;
		my @fl = &{$filter}($_);
		if($fl[0]) {
			$returnage->{$fl[0]} = $fl[1];
		} # end if
	} # end while

	return $returnage;
} # end readfile_filtered_ashashref

sub readline_filtered {	# Runs a readline() though a user-defined sub, returns the result or undef
	my $self = shift;
	my $filter = shift;

	$self->warn("readline_filtered called");

	my $rl = $self->readline();
	my $fl = $filter->($rl);

	return( $fl ? $fl : undef );
} # end readline_filtered

sub _close {	# closes the file and does sanity checking if it's a writer and a real file

	my $self = shift;

	$self->warn("Closing");
	if($self->{'NOTAFILE'} < 1) {
		if(($self->{'WRITER'} > 0) && ($^O !~ m/Win32/)) {
			$self->warn("Looks like we have a writer!");
			$self->warn("Writer wanted to write $self->{'EXPECTEDBYTES'}, got $self->{'WROTEBYTES'}");

			# This is disabled on Windows - it seems to
			# have problems with flushing stuff to disk
			# in time for the check to work.  :(

			if($self->{'EXPECTEDBYTES'} > $self->{'WROTEBYTES'}) {
				$self->process_bang("While writing to '" . $self->fullpath . "': Expected to write $self->{'EXPECTEDBYTES'} chars, but only found $self->{'WROTEBYTES'}.", "", $self->tracer());
			} elsif($self->{'EXPECTEDBYTES'} < $self->{'WROTEBYTES'}) {
				$self->warn("Expected to write out $self->{'EXPECTEDBYTES'}, but only wrote out $self->{'WROTEBYTES'} - this is NON FATAL.");
			} elsif(($self->{'WROTEBYTES'} == 0) && ($self->{'EXPECTEDBYTES'} > 0)) {
				$self->process_bang("While writing to '" . $self->fullpath . "': Expected to write $self->{'EXPECTEDBYTES'} chars, but only found $self->{'WROTEBYTES'}.", "", $self->tracer());
			} # end if

			$self->warn("Got through the sanity checks.  Let's do a stat()");
			my @status;
			my $waitfor;
			WAITER: while($waitfor < 8) {
 				@status = stat($self->{'FULLPATH'});
				last WAITER if(($status[7] ne '') && ($status[7] != 0));
				select(undef, undef, undef, 0.25);
				$waitfor++;
			} # end while
			$self->warn("Stat: @status");

			if(@status) {
				if($status[7] < $self->{'WROTEBYTES'}) {
					$self->process_bang("While writing to '" . $self->fullpath . "': Expected to write $self->{'WROTEBYTES'} chars, but only found $status[7].", "", $self->tracer());
				} elsif($status[7] > $self->{'WROTEBYTES'}) {
					$self->warn("File on disk is $status[7], but I only wrote out $self->{'WROTEBYTES'} (Am I on Win32?)");
				} # end if
			} else {
				$self->process_bang("While writing to '" . $self->fullpath . "': The file I just wrote did not appear on disk within two seconds.", "", $self->tracer());
			} # end if

			$self->warn("Yay!  We got through the stat and the sanity checks.  Let's finish things up.");
		} else {
			$self->warn("Not a writer (or on Win32) - skipping file tests");
		} # end if

		# We can no longer die on !close, sometimes there are really odd
		# errors on Win32...
		CORE::close($self->{'HANDLE'});# or die "Can't close self: $!";
		$self->warn("Closed");
	} else {
		# placeholder for $self->{'CLOSEHANDLER'}
	} # end if notafile
	return 1;
} # end close




sub _open {	# generic _open handler
	my $self = shift;

	my $path = $_[0];
	if($path) {	# alter the path to the files as requird
		$path .= "/" if(($path !~ m!/$!) && ($_[2] !~ m!^/!));
	}

	# define buffered open routines if user has turned buffering on
	$self->_open_buffered() if $self->{'BUFFER'} > 0;

	# The UNBUFFERED print handler
	$self->{'PRINTHANDLER'} = sub {
		my $self = shift;
		my $handle = $self->{'HANDLE'};
		my $to = join("", @_);
		$self->{'WRITER'}++;
		$! = undef;
		my $returned = syswrite($handle, $to, length($to), 0);
		if(!defined $returned) {
			$self->process_bang("Syswrite failed for '" . $self->fullpath . "'.", $!, $self->tracer());
		} elsif($returned < length($to)) {
			$self->process_bang("Syswrite failed for '" . $self->fullpath . "' (wrote $returned, expected " . length($to) . ".", $!, $self->tracer());
		}
		$self->{'EXPECTEDBYTES'} += length($to);
		$self->{'WROTEBYTES'} += $returned;
		$self->warn("Wrote out $returned bytes.");
	} unless exists($self->{'PRINTHANDLER'});

	# The UNBUFFERED readline handler
	$self->{'READLINEHANDLER'} = sub {
		my $self = shift;
		my $handle = $self->{'HANDLE'};
		my ($to, $this_to);
		READCHAR: while(sysread($handle, $this_to, 1) > 0) {
			if($this_to eq "\n") {
				$to .= $this_to;
				return $to;
			} elsif($this_to eq "\r") {	# ignore DOS newlines
				next READCHAR;
			} else {
				$to .= $this_to;
			}
		}
		return $to;
	} unless exists($self->{'READLINEHANDLER'});

	# The UNBUFFERED readfile handler
	$self->{'READFILEHANDLER'} = sub {
		my $self = shift;
		my $handle = $self->{'HANDLE'};
		my($to, $this_to);
		READCHAR: while(sysread($handle, $this_to, 1) > 0) {
			if($this_to eq "\r") {	# ignore DOS newlines
				next READCHAR;
			}
			$to .= $this_to;
		} # yes, we need to read one byte at a time...
		return $to;
	} unless exists($self->{'READFILEHANDLER'});

	# The UNBUFFERED truncate handler
	$self->{'TRUNCATEHANDLER'} = sub {
		my $self = shift;
		return if $self->{'NOTAFILE'} > 0;
		$self->warn("Truncated");
		my $returned = sysseek($self->{'HANDLE'}, 0, 0);
		#my $returned = seek($self->{'HANDLE'}, 0, 0);
		unless($returned eq '0 but true') { $self->process_bang("Sysseek failed for '" . $self->fullpath . "' ($returned).", $!, $self->tracer()); }
		CORE::truncate($self->{'HANDLE'}, 0) or $self->process_bang("Truncate failed for '" . $self->fullpath . "'.", $!, $self->tracer());
		return;
	} unless exists($self->{'TRUNCATEHANDLER'});

	my $handle = $self->{'HANDLE'};

	$self->warn("About to open file as $_[1]");

	$self->{'READMODE'} = $_[1];

	# these routines are used for both buffered and unbuffered output
	if($_[1] eq 'readonly') {
		sysopen($handle, "$path$_[2]", O_RDONLY | O_CREAT, SIXSIXSIX) or $self->process_bang("Can't open '$path$_[2]'.", $!, $self->tracer());
		flock($handle, LOCK_SH);
		$self->{'LOCKMODE'} = 'SH';
	} elsif($_[1] eq 'readonlync') {
		return undef unless -e "$path$_[2]";
		sysopen($handle, "$path$_[2]", O_RDONLY) or $self->process_bang("Can't open '$path$_[2]'.", $!, $self->tracer());
		flock($handle, LOCK_SH);
		$self->{'LOCKMODE'} = 'SH';
	} elsif($_[1] eq 'writeonly') {
		sysopen($handle, "$path$_[2]", O_WRONLY | O_CREAT, SIXSIXSIX) or $self->process_bang("Can't open '$path$_[2]'.", $!, $self->tracer());
		flock($handle, LOCK_EX);
		$self->{'LOCKMODE'} = 'EX';
	} elsif($_[1] eq 'readwrite') {
		sysopen($handle, "$path$_[2]", O_RDWR | O_CREAT, SIXSIXSIX) or $self->process_bang("Can't open '$path$_[2]'.", $!, $self->tracer());
		flock($handle, LOCK_SH);
		$self->{'LOCKMODE'} = 'SH';
	} elsif($_[1] eq 'readwrite2') {	# a readwrite that opens in EX mode
		sysopen($handle, "$path$_[2]", O_RDWR | O_CREAT, SIXSIXSIX) or $self->process_bang("Can't open '$path$_[2]'.", $!, $self->tracer());
		flock($handle, LOCK_EX);
		$self->{'LOCKMODE'} = 'EX';
	} elsif($_[1] eq 'writeappend') {
		sysopen($handle, "$path$_[2]", O_WRONLY | O_CREAT | O_APPEND, SIXSIXSIX) or $self->process_bang("Can't open '$path$_[2]'.", $!, $self->tracer());
		flock($handle, LOCK_EX);
		$self->{'LOCKMODE'} = 'EX';
	} # end if

	my $oldhandle = select($handle);  $| = 1; select($oldhandle);	# autoflush

	$self->{'FULLPATH'} = "$path$_[2]";
	$self->warn("Opened file as $_[1]");

	return 1;
} # end _open



sub _open_cache {	# handles files in the cache directory
	my $self = shift;

	my $path = $self->{'CONFIG'}->{'NonCGIPath'} . "/cache-" .
		$self->{'CONFIG'}->{'cache_pw'};
	$self->_open($path, @_);

	return 1;
} # end _open_cache




sub _open_lock {	# handles lock & log files exclusively
	my $self = shift;

	$self->{'IMALOCK'}++;

	my $path = $self->{'CONFIG'}->{'NonCGIPath'} . "/cache-" .
		$self->{'CONFIG'}->{'cache_pw'} . "/lock";
	$self->_open($path, @_);

	return 1;
} # end _open_lock



sub _open_thread {	# handles just opening thread data files
	my $self = shift;

	my $path = "";
	$self->_open($path, @_);

	return 1;
} # end _open_thread



sub _open_forumd {	# handles forum metadata (.threads, .file, f_t_d, etc)
	my $self = shift;

	my $path = "";
	$self->_open($path, @_);

	return 1;
} # end _open_forumd



sub _open_member {	# handles only member data
	my $self = shift;

	my $path = $self->{'CONFIG'}->{'MembersPath'};
	$self->_open($path, @_);

	return 1;
} # end _open_member




sub _open_file {	# generic file opener
	my $self = shift;

	my $path = "";
	$self->_open($path, @_);

	return 1;
} # end _open_file




sub _open_index {	# 6.2.2 search index
	my $self = shift;

	# for search indexes, do *NOT* use buffering!
#	$self->{'BUFFER'} = 0;
	$self->_open("", @_);

	return 1;
} # end _open_index




sub warn {	# sends warnings to the FileHandler parent to figure out
	my $self = shift;
	return unless $self->{'DEBUG'} > 0;
	my @pass = @_;
	foreach(@pass) {
		$_ = "    (filehandle:" . $self->{'UNIQ'} . ") " . $self->{'FULLPATH'} . ": " . $_;
	} # end foreach
	$self->{'PARENT'}->warn(@pass);
	return;
} # end warn


# I don't need to explain these three...
sub fullpath {
	return $_[0]->{'FULLPATH'};
} # end fullpath
sub readmode {
	return $_[0]->{'READMODE'};
} # end fullpath
sub lockmode {
	return $_[0]->{'LOCKMODE'};
} # end fullpath
sub uniq {
	return $_[0]->{'UNIQ'};
} # end fullpath




sub _open_buffered {	# the special BUFFERED opening handlers
	my $self = shift;
	return unless $self->{'BUFFER'} > 0;

	$self->warn("USING BUFFERED I/O!  ARE YOU NUTS?!");

	# the BUFFERED print handler
	$self->{'PRINTHANDLER'} = sub {
		my $self = shift;
		my $toprint = join("", @_);
		my $handle = $self->{'HANDLE'};
		$self->{'WRITER'}++;
		my $returned = print $handle $toprint;
		my $lengtha = length($toprint);
		if((!defined $returned) || ($returned < 1)) {
			$self->process_bang("Could not perform buffered print to '" . $self->fullpath() . "'.", $!, $self->tracer());
			#die "Could not perform buffered print() to '" . $self->fullpath() . "', error returned by OS is: '$!', length of print requested is: '$lengtha', backtrace is: " . $self->tracer();
		}
		$self->{'EXPECTEDBYTES'} += $lengtha;
		$self->{'WROTEBYTES'} += $lengtha;
		$self->warn("Wrote out $lengtha bytes.");
	} unless exists($self->{'PRINTHANDLER'});

	# the BUFFERED readline handler
	$self->{'READLINEHANDLER'} = sub {
		my $self = shift;
		my $handle = $self->{'HANDLE'};
		return <$handle>;
	} unless exists($self->{'READLINEHANDLER'});

	# the BUFFERED readfile handler
	$self->{'READFILEHANDLER'} = sub {
		my $self = shift;
		my $handle = $self->{'HANDLE'};
		local $/ = undef;	#SLURP!
		return <$handle>;
	} unless exists($self->{'READFILEHANDLER'});

	# the BUFFERED truncate handler
	$self->{'TRUNCATEHANDLER'} = sub {
		my $self = shift;
		return if $self->{'NOTAFILE'} > 0;
		$self->warn("Truncated");
		#my $returned = sysseek($self->{'HANDLE'}, 0, 0);
		my $returned = seek($self->{'HANDLE'}, 0, 0);
		#die "sysseek failed: $! ($returned)" unless $returned eq '0 but true';
		CORE::truncate($self->{'HANDLE'}, 0) or $self->process_bang("Can't truncate '" . $self->fullpath() . "'.", $!, $self->tracer());
		return;
	} unless exists($self->{'TRUNCATEHANDLER'});

	return;
} # end sub


sub tracer {	# tracer, copied here so we don't need to rely on ubb_lib
	my $self = shift;
	my $string = "";
	my $i = 0;
	while(@_ = caller($i)) {
		my $file = $_[1];
		$file =~ s!(.*)(\\|/)([^/\\]+)$!$3!;
		$string .= "\n Backtrace: $file:$_[2] -> sub $_[3] ";
		$i++;
	}
	return $string . "\n";
} # end tracer

sub DESTROY {
	my $self = shift;
	$self->warn("I HAVE JUST BEEN DESTROYED");
} # end DESTROY

1800 + 555.1234 == 2355.1234;
# $Id: ubb_lib_filehandle.cgi,v 1.13 2002/04/18 22:27:17 cvscapps Exp $
