
# Data::ThatWhichDumps
# A pseudo-replacement for Data::Dumper in 5.004

# For very simple complex data structures only.
# Based heavily on work from Advanced Perl Programming.

package Data::ThatWhichDumps;

sub new {
	my $protoobj = shift;
	my $class    = ref($protoobj) || $protoobj;
	my $self     = { NAMES => $_[1], VALUES => $_[0] };
	die "Usage: new Data::ThatWhichDumps(\@values, \@names)\n" unless ((ref($_[0]) eq "ARRAY") && (ref($_[1]) eq "ARRAY"));
	bless($self, $class);
	return $self;
} # end new

sub Dump {
	die "Dump() is the OO method.  Use Dumper()!\n" unless ref($_[0]) =~ m/Data::ThatWhichDumps/;
	my $self = shift;
	my $len = scalar(@{$self->{VALUES}}) - 1;
	my $string = "";
	foreach my $one (0 .. $len) {
		my $sym = substr($self->{NAMES}->[$one], 0, 1);
		$sym = ($sym =~ m/\w/ ? '$' : $sym);  # make it a ref / scalar if it's not anything valid
		if($sym eq '$') {	# ref / scalar
			$string .= $self->_recurse($self->{NAMES}->[$one], undef, undef, $self->{VALUES}->[$one]);
		} elsif($sym eq '@') {	# array
			$string .= $self->_recurse($self->{NAMES}->[$one], "(", ");\n", $self->{VALUES}->[$one]);
		} elsif($sym eq '%') {	# hash
			$string .= $self->_recurse($self->{NAMES}->[$one], "(", ");\n", $self->{VALUES}->[$one]);
		} elsif($sym eq '*') {	# a typeglob.. yeesh...
			die "I don't do typeglobs, use the real symbol";
		} else {		# something else
			die "I don't know what to do with a '$sym'";
		} # end if
	} # end foreach

	return $string;
} # end Dump

sub Dumper {
	die "Dumper() is the non-OO method.  Use Dump()!\n" if ref($_[0]) =~ m/Data::ThatWhichDumps/;
	my $num = scalar(@_);
	my @names;
	foreach(1 .. $num) { push(@names, '$VAR' . $_) }
	my $s = new Data::ThatWhichDumps(\@_, \@names);
	return $s->Dump();
} # end Dumper

sub import {
	return undef;
} # end import

sub _recurse {
	my($self, $name, $openp, $closep, $var) = @_;
	my($iso, $isc);
	$self->{level} = 0;
	$self->{output} = "";
	my $refer = ref($var);

	if($refer) {
		if($refer eq "ARRAY") {
			($iso, $isc) = ("[", "],");
		} elsif($refer eq "HASH") {
			($iso, $isc) = ("{", "},");
		} else {	# ...
			($iso, $isc) = ("(", "),");
		} # end if
		$self->_process_ref($var);
	} else {
		$self->_process_scalar($var);
	} # end if

	my $string = $self->{output};
	$openp = ($openp ? $openp : $iso);
	my $isd = $isc;
	$isd =~ s/\,/\;/;
	$closep = ($closep ? $closep : $isd);

	$iso = quotemeta($iso);
	$isc = quotemeta(reverse $isc);
	$closep = reverse $closep;
	$string =~ s/  $iso/$openp/;

	my $v = reverse $string;
	$v =~ s/$isc/$closep/;
	$string = reverse $v;

	$string =~ s/\,(\s*)$/\;$1/;

	return "$name = $string";
} # end _start

sub _process_scalar {
	my $self = shift;
	++$self->{level};
	$self->_output_append ($self->_quotize(join("", @_)) . ",");
	--$self->{level};
} # end _process_scalar

sub _process_ref {
	my $self = shift;
	my $r = shift;

	my $ref_type = ref($r);
	if ($ref_type eq "ARRAY") {	# array
		$self->_process_array($r);
	} elsif ($ref_type eq "SCALAR") {	# scalar
		$self->_process_scalar($$r);
	} elsif ($ref_type eq "HASH") {	# hash
		$self->_process_hash($r);
	} elsif ($ref_type eq "REF") {	# generic ref, so deref it
		++$self->{level};
		$self->_process_ref($$r);
		--$self->{level};
	} elsif(!$ref_type) {		# not a reference, so make it a ref
		$self->_process_ref(\$r);
	} else {			# something else
		$self->_process_scalar("Unknown ref type '$ref_type': $r");
	} # end if
} # end _process_ref

sub _process_array {
	my $self = shift;
	my $r_array = shift;
	++$self->{level};
	$self->_output_append ("[");
	foreach my $var (@$r_array) {
		if (ref($var)) {
			$self->_process_ref($var);
		} else {
			$self->_process_scalar($var);
		} # end if
	} # end foreach
	$self->_output_append ("],");
	--$self->{level};
} # end _process_array

sub _process_hash {
	my $self = shift;
	my $r_hash = shift;

	++$self->{level};
	$self->_output_append ("{");
	while(my($key, $val) = each %{$r_hash}) {
		++$self->{level};
		if (ref($val)) {
			$self->_output_append ( $self->_quotize($key) . " => ");
			$self->_process_ref($val);
		} else {
			$self->_output_append ( $self->_quotize($key) . " => " . $self->_quotize($val) . ",");
		}
		--$self->{level};
	}
	$self->_output_append ("},");
	--$self->{level};
} # end _process_hash

sub _output_append {
	my $self = shift;
	my $spaces = "  " x $self->{level};
	my $st = join("", @_);
	$self->{output} .= "$spaces$st\n";
} # end _output_append

sub _quotize {
	my $self = shift;
	my $string = shift;

	return 'undef' unless defined $string;

	foreach my $char ('!', '~', '#', '%', '^', '&', '/', "'", '"', ',', '.', '|') {
		my $c = quotemeta($char);
		if($string !~ m/$c/) {
			return qq(q$char$string$char);
		} # end if
	} # end foreach

	# if somehow the string contains all of our quotes, try using a paren combo...

	foreach my $char ('(|)', '[|]', '{|}') {
		my($left, $right) = split(/\|/, $char);
		my $l = quotemeta($left); my $r = quotemeta($right);
		if($string !~ m/$l|$r/) {
			return qq(q$left$string$right);
		} # end if
	} # end foreach

	# Hm, so it uses all of our quotes AND all of our parens!?  Wow.

	# Okay, oldest trick in the book....
	$string =~ s/\|/\&\#0124\;/g;
	return qq(q|$string|);
} # end _quotize

1;
# $Id: ubb_lib_dumper.cgi,v 1.8 2002/03/30 00:19:42 cvscapps Exp $
